(* ================================================================== *)
(* == POD 98-99 --                                                    *)
(* ------------------------------------------------------------------ *)
(* == Client/Serveur via Internet : Hello v.03                        *)
(* La version du client avec delai d'attente pour la reponse et       *)
(* acquittement                                                       *)
(* ------------------------------------------------------------------ *)
(*   ocamlc -custom unix.cma hello3.ml -o hello3 -cclib -lunix        *)
(* ================================================================== *)

open Unix
;;

(* == val get_in_addr : string -> Unix.inet_addr                      *)
(* Calcule l'adresse IP du serveur en fonction de son argument        *)
let get_in_addr host = 
 try 
  (* on essaie 'machine.domaine' *)
  (gethostbyname host).h_addr_list.(0)
 with 
  Not_found 
  -> try 
      (* sinon : x.y.z.w *)
      inet_addr_of_string host       
     with
      Failure "inet_addr_of_string" -> failwith("unknown host "^host)
;;

(* == val get_s_port : string -> int                                  *)
(* Calcule le numero de port du service en fonction de son argument   *)
let get_s_port port =
 try int_of_string port with
  Failure("int_of_string") -> failwith("incorrect port number "^port)
;;

(* == val client : unit -> unit                                       *)
(* Le client                                                          *)
let client () =
 (* Si pas assez d'arguments : erreur    *)
 if Array.length Sys.argv < 3 then
   begin
    Printf.eprintf "Usage : hello2 hostname portnumber\n"
   end
 (* sinon *)
 else
  (* Delai maximal d'attente *)
  let timeout = 5 in
  (* Adresse du serveur (1ier argument) *)
  let in_addr = get_in_addr Sys.argv.(1) in
  (* Port de connexion (2nd argument) *)
  let s_port = get_s_port Sys.argv.(2) in
  (* Creation de la socket de service *)
  let sock = socket PF_INET SOCK_STREAM 0 in
    (* Processus de d'attente et traitement de la reponse (le fils) *)
    if fork() = 0 then
     begin
      (* On declenche le compte a rebours *)
      alarm timeout;
      (* Connexion *)
      connect sock (ADDR_INET(in_addr, s_port));
      (* Lecture et affichage de la reponse *)
      let r = input_line (in_channel_of_descr sock) in
        (Printf.printf "(%d) %s\n" (String.length r) r;
         flush (out_channel_of_descr stdout));
      (* Emission de l'acquitement *)
      write sock "OK" 0 2;
      (* Fermeture de la socket *)
      close sock;
      (* Fin du traitement *)
      exit 0
     end
    else (* Le pere *)
     begin
      (* Fermeture de la socket *)
      close sock;
      (* Attente et analyse fin du fils *)
      (match wait() with
        (* Fin normale *)
        (_,WEXITED 0) -> ()
         (* Fin hors delai *)
      |(_,WSIGNALED 14) -> Printf.printf"Terminated after timeout %d sec.\n" timeout
        (* Autres cas *)
       |_ -> Printf.eprintf"Unexpected terminaison\n")
     end
;;

(* == Programme principal *)
try
 client()
with
 Unix_error(c,f,x)
 -> Printf.eprintf "\n %s in %s %s\n" (error_message c) f x
;;
