(* ================================================================== *)
(* == POD 98-99 --                                                    *)
(* ------------------------------------------------------------------ *)
(* == Client/Serveur via Internet : Hello v.03                        *)
(* La version du serveur avec attente d'acquitement et ecriture d'une *)
(* trace dans un fichier 'hello.log'                                  *)
(* ------------------------------------------------------------------ *)
(*   ocamlc -custom unix.cma hello3d.ml -o hello3d -cclib -lunix      *)
(* ================================================================== *)

open Unix

(* == val get_s_port : string -> int                                  *)
(* Calcule le numero de port du service en fonction de son argument   *)
let get_s_port p_num =
 try int_of_string p_num with
  Failure("int_of_string") -> failwith("incorrect port number "^p_num)
;;

(* == val get_cl_name : Unix.sockaddr -> string                       *)
(* Extrait le nom du client a partie de sont addresse IP              *)
let get_cl_name = function
 ADDR_UNIX _ -> failwith "INET socket expected"
|ADDR_INET(cl_inet_addr,_) -> (gethostbyaddr cl_inet_addr).h_name
;;

(* == val message : string -> string -> string                        *)
(* Compose le message de reponse a la connection d'un client          *)
let message cl_addr n_con =
 Printf.sprintf "Hello %s (%s)\n" cl_addr n_con
;;

(* == val write_log : string -> unit                                  *)
(* Ecriture d'une ligne dans le fichier trace                         *)
let write_log s =
 let f = open_out_gen [Open_wronly;Open_text;Open_append; Open_creat] (6*64+4*8+4) "hello.log" in
  output_string f s;
  close_out f
;;

(* == val con_line : string -> Unix.tm -> string -> string            *)
(* Construction d'une ligne pour le fichier trace :                   *)
(* Demande de connexion : nom-client date heure numero                *)
let con_line cl_name t_con n_con = 
 Printf.sprintf "%s connection %d/%d/%d %d:%d:%d (%s)\n"
                cl_name t_con.tm_mday t_con.tm_mon t_con.tm_year t_con.tm_hour
                t_con.tm_min t_con.tm_sec n_con
;;
 
(* == val ack_line : string -> Unix.tm -> Unix.tm -> string -> string *)
(* Construction d'une ligne pour le fichier trace :                   *)
(* Acquitement :                                                      *)
(*  nom-client date heure date-demande heure-demande numero-demande   *)
let ack_line cl_name t_ack t_con n_con = 
 Printf.sprintf "%s acknoledgement %d/%d/%d %d:%d:%d [%d/%d/%d %d:%d:%d (%s) request] \n"
                cl_name 
                t_ack.tm_mday t_ack.tm_mon t_ack.tm_year t_ack.tm_hour 
                t_ack.tm_min t_ack.tm_sec
                t_con.tm_mday t_con.tm_mon t_con.tm_year t_con.tm_hour
                t_con.tm_min t_con.tm_sec n_con 
;;

(* == val mk_n_con : float -> string                                  *)
(* Construction d'un numero de demande de connexion (decimales du     *)
(* temps systeme                                                      *)
let mk_n_con t =
 let t = string_of_float t in
 let i = String.index t '.' in
 let l = (String.length t)-i-1 in
   String.sub t (i+1) l
;;
 
(* == val server : unit -> unit                                       *)
(* Le serveur                                                         *)
let server () =
 (* Si pas d'argument : erreur    *)
 if Array.length Sys.argv < 2 then
   begin
    Printf.eprintf "Usage : hello3d portnumber\n"
   end
 (* sinon *)
 else
  (* Lecture du numero de port du service *)
  let port = get_s_port Sys.argv.(1) in
  (* Creation de la socket de service *)
  let sock = socket PF_INET SOCK_STREAM 0 in
  (* Adresse du serveur *)
  let h_addr = (gethostbyname (gethostname())).h_addr_list.(0) in
  let sock_addr = ADDR_INET(h_addr, port) in
  let sock_addr = ADDR_INET(h_addr, port) in
   begin
    (* Liaison socket/adresse *)
    bind sock sock_addr;
    (* Mise en attente de connection *)
    listen sock 5;
    (* Boucle de service *)
    while true do
     (* On accepte une connexion *)
     let (cl_sock, cl_sock_addr) = accept sock in
      (* On cree un processus de traitement de la connexion (le fils) *)
      if fork() == 0 then
       begin
        (* Hack Unix :-) *)
        if fork()<>0 then exit 0;
        (* Nom du client *)
        let cl_name = get_cl_name cl_sock_addr in
        (* Temps (systeme) de connexion *)
        let t_con = Unix.gettimeofday() in 
        (* Numero de connexion *)
        let n_con = mk_n_con t_con in
        (* Date locale de connexion *)
        let t_con = Unix.localtime t_con in 
         (* Emmission de message reponse *)
         (let m = message cl_name n_con in
           write cl_sock m 0 (String.length m));
         (* Ecriture trace connexion *)
         write_log (con_line cl_name t_con n_con);
         (* Lecture de l'acquitement *)
         (let r = String.make 2 '\000' in
           read cl_sock r 0 2);
         (* Ecriture trace acquittement *)
         (let t_ack = Unix.localtime(Unix.gettimeofday()) in 
           write_log (ack_line cl_name t_ack t_con n_con));
         (* Fermeture de la socket client *)
         close cl_sock;
         (* Fin du processus de traitement de connexion *)
         exit 0
       end
      else (* Le pere *)
       (* Attente fin de traitement connexion (par le fils) *)
       let _ = wait() in close cl_sock
     done
    end
;;

(* == Programme principal *)
try
 server()
with
 Unix_error(c,f,x)
 -> Printf.eprintf "\n %s in %s %s\n" (error_message c) f x
|Failure x
 -> Printf.eprintf "\n %s \n" x
;;


